unit DelphiSupport;

{$I Defines.inc}

interface

uses
  Windows,
  SysUtils,
  Classes,
  ActiveX,
  ComObj,
  VarUtils,
  SampleDLLHeaders;

type
  TDebugName = String[99];

  TBaseObject = class(TInterfacedObject, IGetImplementation, ISupportErrorInfo)
  strict private
    FName: TDebugName;
    function GetRefCount: Integer;
  strict protected
    procedure SetName(const AName: String);
    function _AddRef: Integer; stdcall;
    function _Release: Integer; stdcall;
  strict protected
    function InterfaceSupportsErrorInfo(const iid: TIID): HResult; stdcall;
    function GetSelf: Pointer;
  public
    constructor Create; virtual;
    procedure BeforeDestruction; override;
    property RefCount: Integer read GetRefCount;
    function SafeCallException(ExceptObject: TObject; ExceptAddr: Pointer): HResult; override;
    property DebugName: TDebugName read FName;
  end;

  TMalloc = class(TBaseObject, IMalloc, ISupportErrorInfo)
  strict protected
    function Alloc(cb: Longint): Pointer; stdcall;
    function Realloc(pv: Pointer; cb: Longint): Pointer; stdcall;
    procedure Free(pv: Pointer); stdcall;
    function GetSize(pv: Pointer): Longint; stdcall;
    function DidAlloc(pv: Pointer): Integer; stdcall;
    procedure HeapMinimize; stdcall;
  public
    constructor Create; override;
  end;

  TNotify = class(TBaseObject, INotify, ISupportErrorInfo)
  strict private
    FNotify: TNotifyEvent;
  strict protected
    procedure Notify; safecall;
  public
    constructor Create(const ANotify: TNotifyEvent); reintroduce;
  end;

  ECheckedInterfacedObjectError = class(Exception);
    ECheckedInterfacedObjectDeleteError = class(ECheckedInterfacedObjectError);
    ECheckedInterfacedObjectDoubleFreeError = class(ECheckedInterfacedObjectError);
    ECheckedInterfacedObjectUseDeletedError = class(ECheckedInterfacedObjectError);

function HandleSafeCallException(ExceptObj: TObject; ErrorAddr: Pointer): HRESULT;

function ObjDescr(const AObj: TObject): String;
function MethodDescr(const AMethod): String;

type
  TInitFunc = procedure(const AOptions: IUnknown);
  TDoneFunc = procedure;
  TInitDoneFunc = record
    Init: TInitFunc;
    Done: TDoneFunc;
  end;

procedure RegisterInitFunc(const AInitProc: TInitFunc; const ADoneFunc: TDoneFunc = nil);

var
  LoadLibraryEx: function(lpFileName: PChar; Reserved: THandle; dwFlags: DWORD): HMODULE; stdcall;
  SetDllDirectory: function(lpPathName: PChar): BOOL; stdcall;
  SetSearchPathMode: function(Flags: DWORD): BOOL; stdcall;
  AddDllDirectory: function(Path: PWideChar): Pointer; stdcall;
  RemoveDllDirectory: function(Cookie: Pointer): BOOL; stdcall;
  SetDefaultDllDirectories: function(DirectoryFlags: DWORD): BOOL; stdcall;

function IsKB959426Installed: Boolean;
function IsKB2533623Installed: Boolean;

const
  // LoadLibraryEx:
  DONT_RESOLVE_DLL_REFERENCES         = $00000001;
  LOAD_LIBRARY_AS_DATAFILE            = $00000002;
  LOAD_WITH_ALTERED_SEARCH_PATH       = $00000008;
  LOAD_IGNORE_CODE_AUTHZ_LEVEL        = $00000010;
  LOAD_LIBRARY_AS_IMAGE_RESOURCE      = $00000020;
  LOAD_LIBRARY_AS_DATAFILE_EXCLUSIVE  = $00000040;
  LOAD_LIBRARY_SEARCH_DLL_LOAD_DIR    = $00000100;
  LOAD_LIBRARY_SEARCH_APPLICATION_DIR = $00000200;
  LOAD_LIBRARY_SEARCH_USER_DIRS       = $00000400;
  LOAD_LIBRARY_SEARCH_SYSTEM32        = $00000800;
  LOAD_LIBRARY_SEARCH_DEFAULT_DIRS    = $00001000;

  // SetSearchPathMode:
  BASE_SEARCH_PATH_ENABLE_SAFE_SEARCHMODE  = $00000001;
  BASE_SEARCH_PATH_PERMANENT               = $00008000;
  BASE_SEARCH_PATH_DISABLE_SAFE_SEARCHMODE = $00010000;

  // SetDefaultDllDirectories:
  // LOAD_LIBRARY_SEARCH_APPLICATION_DIR = $00000200;
  // LOAD_LIBRARY_SEARCH_USER_DIRS       = $00000400;
  // LOAD_LIBRARY_SEARCH_SYSTEM32        = $00000800;
  // LOAD_LIBRARY_SEARCH_DEFAULT_DIRS    = $00001000;

function LoadDLL(const ADLLName: UnicodeString;
  ErrorMode: UINT = SEM_NOOPENFILEERRORBOX): HMODULE;

implementation

function SetErrorInfo(const ErrorCode: HRESULT; const ErrorIID: TGUID;
  const Source, Description, HelpFileName: WideString;
  const HelpContext: Integer): HRESULT;
var
  CreateError: ICreateErrorInfo;
  ErrorInfo: IErrorInfo;
begin
  Result := E_UNEXPECTED;
  if Succeeded(CreateErrorInfo(CreateError)) then
  begin
    CreateError.SetGUID(ErrorIID);
    if Source <> '' then
      CreateError.SetSource(PWideChar(Source));
    if HelpFileName <> '' then
      CreateError.SetHelpFile(PWideChar(HelpFileName));
    if Description <> '' then
      CreateError.SetDescription(PWideChar(Description));
    if HelpContext <> 0 then
      CreateError.SetHelpContext(HelpContext);
    if ErrorCode <> 0 then
      Result := ErrorCode;
    if CreateError.QueryInterface(IErrorInfo, ErrorInfo) = S_OK then
      ActiveX.SetErrorInfo(0, ErrorInfo);
  end;
end;

function GetErrorInfo(out ErrorIID: TGUID; out Source, Description, HelpFileName: WideString; out HelpContext: Longint): Boolean;
var
  ErrorInfo: IErrorInfo;
begin
  if ActiveX.GetErrorInfo(0, ErrorInfo) = S_OK then
  begin
    ErrorInfo.GetGUID(ErrorIID);
    ErrorInfo.GetSource(Source);
    ErrorInfo.GetDescription(Description);
    ErrorInfo.GetHelpFile(HelpFileName);
    ErrorInfo.GetHelpContext(HelpContext);
    Result := (Description <> '') or (Source <> '') or (not CompareMem(@ErrorIID, @GUID_NULL, SizeOf(ErrorIID)));
  end
  else
  begin
    FillChar(ErrorIID, SizeOf(ErrorIID), 0);
    Source := '';
    Description := '';
    HelpFileName := '';
    HelpContext := 0;
    Result := False;
  end;
end;

function SDBMHash(const AData: Pointer; const ADataSize: Cardinal): Cardinal; overload;
var
  P: PByte;
  X: Integer;
begin
  Result := 0;

  P := AData;
  if (P = nil) or (ADataSize = 0) then
    Exit;

  {$R-}
  for X := ADataSize - 1 downto 0 do
  begin
    Result := P^ + (Result shl 6) + (Result shl 16) - Result;
    Inc(P);
  end;
  {$R+}
end;

function SDBMHash(const AData: RawByteString): Cardinal; overload;
begin
  Result := SDBMHash(Pointer(AData), Length(AData));
end;

function Hash(const AData: String): Word;
var
  Code: Cardinal;
begin
  Code := SDBMHash(UTF8Encode(AData));
  Result := Word(Code);
end;

const
  cDelphiException = $0EEDFADE;

function Exception2HRESULT(const E: TObject): HRESULT;

  function NTSTATUSFromException(const E: EExternal): DWORD;
  begin
    if E.InheritsFrom(EDivByZero) then
      Result := STATUS_INTEGER_DIVIDE_BY_ZERO
    else
    if E.InheritsFrom(ERangeError) then
      Result := STATUS_ARRAY_BOUNDS_EXCEEDED
    else
    if E.InheritsFrom(EIntOverflow) then
      Result := STATUS_INTEGER_OVERFLOW
    else
    if E.InheritsFrom(EInvalidOp) then
      Result := STATUS_FLOAT_INVALID_OPERATION
    else
    if E.InheritsFrom(EZeroDivide) then
      Result := STATUS_FLOAT_DIVIDE_BY_ZERO
    else
    if E.InheritsFrom(EOverflow) then
      Result := STATUS_FLOAT_OVERFLOW
    else
    if E.InheritsFrom(EUnderflow) then
      Result := STATUS_FLOAT_UNDERFLOW
    else
    if E.InheritsFrom(EAccessViolation) then
      Result := STATUS_ACCESS_VIOLATION
    else
    if E.InheritsFrom(EPrivilege) then
      Result := STATUS_PRIVILEGED_INSTRUCTION
    else
    if E.InheritsFrom(EControlC) then
      Result := STATUS_CONTROL_C_EXIT
    else
    {$WARNINGS OFF}
    if E.InheritsFrom(EStackOverflow) then
    {$WARNINGS ON}
      Result := STATUS_STACK_OVERFLOW
    else
      Result := STATUS_NONCONTINUABLE_EXCEPTION;
  end;

begin
  if E = nil then
    Result := E_UNEXPECTED
  else
  if not E.InheritsFrom(Exception) then
    Result := E_UNEXPECTED
  else
  if E.ClassType = Exception then
    Result := E_FAIL
  else
  if E.InheritsFrom(ESafecallException) then
    Result := E_FAIL
  else
  if E.InheritsFrom(EAssertionFailed) then
    Result := E_UNEXPECTED
  else
  if E.InheritsFrom(EAbort) then
    Result := EAbortRaisedHRESULT
  else
  if E.InheritsFrom(EOutOfMemory) then
    Result := E_OUTOFMEMORY
  else
  if E.InheritsFrom(ENotImplemented) then
    Result := E_NOTIMPL
  else
  if E.InheritsFrom(ENotSupportedException) then
    Result := E_NOINTERFACE
  else
  if E.InheritsFrom(EOleSysError) then
    Result := EOleSysError(E).ErrorCode
  else
  if E.InheritsFrom(ESafeArrayError) then
    Result := ESafeArrayError(E).ErrorCode
  else
  if E.InheritsFrom(EOSError) then
    Result := HResultFromWin32(EOSError(E).ErrorCode)
  else
  if E.InheritsFrom(EExternal) then
    if Failed(HRESULT(EExternal(E).ExceptionRecord.ExceptionCode)) then
      Result := HResultFromNT(Integer(EExternal(E).ExceptionRecord.ExceptionCode))
    else
      Result := HResultFromNT(Integer(NTSTATUSFromException(EExternal(E))))
  else
    Result := MakeResult(SEVERITY_ERROR, FACILITY_ITF, Hash(E.ClassName)) or CUSTOMER_BIT;
end;

function HRESULT2Exception(const E: HRESULT; var ErrorAddr: Pointer): Exception;

  // This is a slightly modified code from SysUtils
  // Unfortunately, this code is not published, so we need to copy it
  // It sets relation between NTStatus system codes and exception classes
  function MapNTStatus(const ANTStatus: DWORD): ExceptClass;
  begin
    case ANTStatus of
      STATUS_INTEGER_DIVIDE_BY_ZERO:
        Result := EDivByZero;
      STATUS_ARRAY_BOUNDS_EXCEEDED:
        Result := ERangeError;
      STATUS_INTEGER_OVERFLOW:
        Result := EIntOverflow;
      STATUS_FLOAT_INEXACT_RESULT,
      STATUS_FLOAT_INVALID_OPERATION,
      STATUS_FLOAT_STACK_CHECK:
        Result := EInvalidOp;
      STATUS_FLOAT_DIVIDE_BY_ZERO:
        Result := EZeroDivide;
      STATUS_FLOAT_OVERFLOW:
        Result := EOverflow;
      STATUS_FLOAT_UNDERFLOW,
      STATUS_FLOAT_DENORMAL_OPERAND:
        Result := EUnderflow;
      STATUS_ACCESS_VIOLATION:
        Result := EAccessViolation;
      STATUS_PRIVILEGED_INSTRUCTION:
        Result := EPrivilege;
      STATUS_CONTROL_C_EXIT:
        Result := EControlC;
      STATUS_STACK_OVERFLOW:
      {$WARNINGS OFF}
        Result := EStackOverflow;
      {$WARNINGS ON}
      else
        Result := EExternal;
    end;
  end;

  function MapException(const ACode: DWORD): ExceptClass;
  begin
    case ACode of
      E_C_AbstractError:               Result := EAbstractError;
      E_C_ArgumentException:           Result := EArgumentException;
      E_C_ArgumentNilException:        Result := EArgumentNilException;
      E_C_ArgumentOutOfRangeException: Result := EArgumentOutOfRangeException;
      E_C_BitsError:                   Result := EBitsError;
      E_C_ClassNotFound:               Result := EClassNotFound;
//      E_C_CodesetConversion:           Result := ECodesetConversion;
      E_C_ComponentError:              Result := EComponentError;
      E_C_ConvertError:                Result := EConvertError;
      E_C_DirectoryNotFoundException:  Result := EDirectoryNotFoundException;
      E_C_External:                    Result := EExternal;
      E_C_ExternalException:           Result := EExternalException;
      E_C_FCreateError:                Result := EFCreateError;
      E_C_FileNotFoundException:       Result := EFileNotFoundException;
      E_C_FilerError:                  Result := EFilerError;
      E_C_FileStreamError:             Result := EFileStreamError;
      E_C_FOpenError:                  Result := EFOpenError;
      E_C_HeapException:               Result := EHeapException;
      E_C_InOutError:                  Result := EInOutError;
      E_C_IntError:                    Result := EIntError;
      E_C_IntfCastError:               Result := EIntfCastError;
      E_C_InvalidCast:                 Result := EInvalidCast;
      E_C_InvalidContainer:            Result := EInvalidContainer;
      E_C_InvalidImage:                Result := EInvalidImage;
      E_C_InvalidInsert:               Result := EInvalidInsert;
      E_C_InvalidOperation:            Result := EInvalidOperation;
      E_C_InvalidOpException:          Result := EInvalidOpException;
      E_C_InvalidPointer:              Result := EInvalidPointer;
      E_C_ListError:                   Result := EListError;
      E_C_MathError:                   Result := EMathError;
      E_C_MethodNotFound:              Result := EMethodNotFound;
      E_C_Monitor:                     Result := EMonitor;
      E_C_MonitorLockException:        Result := EMonitorLockException;
      E_C_NoConstructException:        Result := ENoConstructException;
      E_C_NoMonitorSupportException:   Result := ENoMonitorSupportException;
      E_C_OutOfResources:              Result := EOutOfResources;
      E_C_PackageError:                Result := EPackageError;
      E_C_ParserError:                 Result := EParserError;
      E_C_PathTooLongException:        Result := EPathTooLongException;
      E_C_ProgrammerNotFound:          Result := EProgrammerNotFound;
      E_C_PropReadOnly:                Result := EPropReadOnly;
      E_C_PropWriteOnly:               Result := EPropWriteOnly;
//      E_C_Quit:                        Result := EQuit;
      E_C_RangeError:                  Result := ERangeError;
      E_C_ReadError:                   Result := EReadError;
      E_C_ResNotFound:                 Result := EResNotFound;
      E_C_StreamError:                 Result := EStreamError;
      E_C_StringListError:             Result := EStringListError;
      E_C_VariantError:                Result := EVariantError;
      E_C_WriteError:                  Result := EWriteError;
    else
      Result := Exception;
    end;
  end;

var
  NTStatus: DWORD;
  ErrorIID: TGUID;
  Source: WideString;
  Description: WideString;
  HelpFileName: WideString;
  HelpContext: Integer;
begin
  if GetErrorInfo(ErrorIID, Source, Description, HelpFileName, HelpContext) then
  begin
    if Pointer(StrToInt64Def(Source, 0)) <> nil then
      ErrorAddr := Pointer(StrToInt64(Source));
  end
  else
    Description := SysErrorMessage(DWORD(E));

  if (E = E_FAIL) or (E = E_UNEXPECTED) then
    Result := Exception.Create(Description)
  else
  if E = EAbortRaisedHRESULT then
    Result := EAbort.Create(Description)
  else
  if E = E_OUTOFMEMORY then
  begin
    OutOfMemoryError;
    Result := nil;
  end
  else
  if E = E_NOTIMPL then
    Result := ENotImplemented.Create(Description)
  else
  if E = E_NOINTERFACE then
    Result := ENotSupportedException.Create(Description)
  else
  if HResultFacility(E) = FACILITY_WIN32 then
  begin
    Result := EOSError.Create(Description);
    EOSError(Result).ErrorCode := HResultCode(E);
  end
  else
  if E and FACILITY_NT_BIT <> 0 then
  begin
    // Get exception class by code
    NTStatus := Cardinal(E) and (not FACILITY_NT_BIT);
    Result := MapNTStatus(NTStatus).Create(Description);

    // Create dummy ExceptionRecord just in case
    ReallocMem(Pointer(Result), Result.InstanceSize + SizeOf(TExceptionRecord));
    EExternal(Result).ExceptionRecord := Pointer(NativeUInt(Result) + Cardinal(Result.InstanceSize));
    FillChar(EExternal(Result).ExceptionRecord^, SizeOf(TExceptionRecord), 0);

    EExternal(Result).ExceptionRecord.ExceptionCode := cDelphiException;
    EExternal(Result).ExceptionRecord.ExceptionAddress := ErrorAddr;
  end
  else
  if (E and CUSTOMER_BIT <> 0) and
     (HResultFacility(E) = FACILITY_ITF) and
     CompareMem(@SampleDllIID, @ErrorIID, SizeOf(ErrorIID)) then
    Result := MapException(HResultCode(E)).Create(Description)
  else
    Result := EOleException.Create(Description, E, Source, HelpFileName, HelpContext);
end;

{ TBaseObject }

resourcestring
  rsInvalidDelete  = 'Attempt to delete the %s object while still having reference to it; reference counter: %d';
  rsDoubleFree     = 'Attempt to delete already deleted object %s';
  rsUseDeleted     = 'Attempt to use already deleted object %s';

constructor TBaseObject.Create;
begin
  FName := TDebugName(Format('[$%s] %s', [IntToHex(NativeUInt(Self), SizeOf(Pointer) * 2), ClassName]));
  inherited;
end;

procedure TBaseObject.BeforeDestruction;
begin
  if FRefCount < 0 then
    raise ECheckedInterfacedObjectDoubleFreeError.CreateFmt(rsDoubleFree, [String(FName)])
  else
  if FRefCount <> 0 then
    raise ECheckedInterfacedObjectDeleteError.CreateFmt(rsInvalidDelete, [String(FName), FRefCount]);
  inherited;
  FRefCount := -1;
end;

procedure TBaseObject.SetName(const AName: String);
begin
  FillChar(FName, SizeOf(FName), 0);
  FName := TDebugName(AName);
end;

function TBaseObject._AddRef: Integer;
begin
  if FRefCount < 0 then
    raise ECheckedInterfacedObjectUseDeletedError.CreateFmt(rsUseDeleted, [String(FName)]);
  Result := InterlockedIncrement(FRefCount);
end;

function TBaseObject._Release: Integer;
begin
  Result := InterlockedDecrement(FRefCount);
  if Result = 0 then
    Destroy;
end;

function TBaseObject.GetRefCount: Integer;
begin
  if FRefCount < 0 then
    Result := 0
  else
    Result := FRefCount;
end;

function TBaseObject.InterfaceSupportsErrorInfo(const iid: TIID): HResult;
begin
  Result := S_OK;
end;

function TBaseObject.GetSelf: Pointer;
begin
  Result := Pointer(Self);
end;

function TBaseObject.SafeCallException(ExceptObject: TObject; ExceptAddr: Pointer): HResult;
begin
  Result := HandleSafeCallException(ExceptObject, ExceptAddr);
end;

{ TMalloc }

constructor TMalloc.Create;
begin
  inherited;
  SetName('DLL Allocator');
end;

function TMalloc.Alloc(cb: Integer): Pointer;
begin
  try
    Result := AllocMem(cb);
  except
    Result := nil;
  end;
end;

function TMalloc.Realloc(pv: Pointer; cb: Integer): Pointer;
begin
  try
    ReallocMem(pv, cb);
    Result := pv;
  except
    Result := nil;
  end;
end;

procedure TMalloc.Free(pv: Pointer);
begin
  try
    if pv <> nil then
      FreeMem(pv);
  except
  end;
end;

function TMalloc.DidAlloc(pv: Pointer): Integer;
begin
  Result := -1;
end;

function TMalloc.GetSize(pv: Pointer): Longint;
begin
  Result := -1;
end;

procedure TMalloc.HeapMinimize;
begin
  // does nothing
end;

var
  GInitDoneFuncs: array of TInitDoneFunc;

procedure RegisterInitFunc(const AInitProc: TInitFunc; const ADoneFunc: TDoneFunc);
begin
  SetLength(GInitDoneFuncs, Length(GInitDoneFuncs) + 1);
  GInitDoneFuncs[High(GInitDoneFuncs)].Init := AInitProc;
  GInitDoneFuncs[High(GInitDoneFuncs)].Done := ADoneFunc;
end;

procedure RunDLLInitFuncs(const AOptions: IUnknown); safecall;
var
  X: Integer;
begin
  for X := 0 to High(GInitDoneFuncs) do
    if Assigned(GInitDoneFuncs[X].Init) then
      GInitDoneFuncs[X].Init(AOptions);
end;

procedure RunDLLDoneFuncs;
var
  X: Integer;
begin
  for X := 0 to High(GInitDoneFuncs) do
    if Assigned(GInitDoneFuncs[X].Done) then
      GInitDoneFuncs[X].Done;
end;

function IsKB959426Installed: Boolean; // XP SP2
begin
  Result := Assigned(SetSearchPathMode);
end;

function IsKB2533623Installed: Boolean; // Vista/7 with KB2533623
begin
  Result := Assigned(AddDllDirectory);
end;

procedure LoadFuncs;
var
  LibKernel32: HMODULE;
begin
  LibKernel32 := GetModuleHandle(kernel32);
  LoadLibraryEx := GetProcAddress(LibKernel32, {$IFDEF UNICODE}'LoadLibraryExW'{$ELSE}'LoadLibraryExA'{$ENDIF});
  SetDllDirectory := GetProcAddress(LibKernel32, {$IFDEF UNICODE}'SetDllDirectoryW'{$ELSE}'SetDllDirectoryA'{$ENDIF});
  SetSearchPathMode := GetProcAddress(LibKernel32, 'SetSearchPathMode');
  AddDllDirectory := GetProcAddress(LibKernel32, 'AddDllDirectory');
  RemoveDllDirectory := GetProcAddress(LibKernel32, 'RemoveDllDirectory');
  SetDefaultDllDirectories := GetProcAddress(LibKernel32, 'SetDefaultDllDirectories');
end;

function LoadDLL(const ADLLName: UnicodeString; ErrorMode: UINT): HMODULE;
var
  DLLPath: String;
  OldDir: String;
  OldPath: UnicodeString;
  OldMode: UINT;
  {$IFDEF WIN32}
  FPUControlWord: Word;
  {$ENDIF}
  {$IFDEF WIN64}
  FPUControlWord: Word;
  {$ENDIF}
begin
  OldDir := GetCurrentDir;
  OldPath := GetEnvironmentVariable('PATH');
  OldMode := SetErrorMode(ErrorMode);
  try
    DLLPath := ExtractFilePath(ADLLName);
    SetEnvironmentVariableW('PATH', PWideChar(UnicodeString(DLLPath) + ';' + OldPATH));
    SetCurrentDir(DLLPath);

    {$IFDEF WIN32}
    asm
      FNSTCW  FPUControlWord
    end;
    {$ENDIF}
    {$IFDEF WIN64}
    FPUControlWord := Get8087CW();
    {$ENDIF}
    try

      if IsKB2533623Installed then
        Result := LoadLibraryExW(PWideChar(ADLLName), 0, LOAD_WITH_ALTERED_SEARCH_PATH)
      else
      if Assigned(SetDllDirectory) then
      begin
        SetDllDirectory(PWideChar(DLLPath));
        try
          Result := LoadLibraryW(PWideChar(ADLLName));
          Win32Check(Result <> 0);
        finally
          SetDllDirectory(nil);
        end;
      end
      else
        Result := LoadLibraryW(PWideChar(ADLLName));
      Win32Check(Result <> 0);

    finally
      {$IFDEF WIN32}
      asm
        FNCLEX
        FLDCW FPUControlWord
      end;
      {$ENDIF}
      {$IFDEF WIN64}
      TestAndClearFPUExceptions(0);
      Set8087CW(FPUControlWord);
      {$ENDIF}
    end;
  finally
    SetErrorMode(OldMode);
    SetEnvironmentVariableW('PATH', PWideChar(OldPATH));
    SetCurrentDir(OldPath);
  end;
  SetLastError(0);
end;

function HandleSafeCallException(ExceptObj: TObject; ErrorAddr: Pointer): HRESULT;
var
  ErrorMessage: String;
  HelpFileName: String;
  HelpContext: Integer;
begin
  if ExceptObj is Exception then
    ErrorMessage := Exception(ExceptObj).Message
  else
    ErrorMessage := SysErrorMessage(DWORD(E_FAIL));
  if ExceptObj is EOleException then
  begin
    HelpFileName := EOleException(ExceptObj).HelpFile;
    HelpContext := EOleException(ExceptObj).HelpContext;
  end
  else
  begin
    HelpFileName := '';
    if ExceptObj is Exception then
      HelpContext := Exception(ExceptObj).HelpContext
    else
      HelpContext := 0;
  end;

  Result := SetErrorInfo(Exception2HRESULT(ExceptObj), SampleDllIID,
    '$' + IntToHex(NativeUInt(ErrorAddr), SizeOf(ErrorAddr) * 2), ErrorMessage,
    HelpFileName, HelpContext);
end;

procedure RaiseSafeCallException(ErrorCode: HResult; ErrorAddr: Pointer);
var
  E: Exception;
begin
  E := HRESULT2Exception(ErrorCode, ErrorAddr);
  raise E at ErrorAddr;
end;

//_______________________________________________________________

{$IFDEF WIN32}
// Fix for the https://quality.embarcadero.com/browse/RSP-24652

type
  PExceptionRecord = ^TExceptionRecord;
  TExceptionRecord = record
    ExceptionCode: Cardinal;
    ExceptionFlags: Cardinal;
    ExceptionRecord: PExceptionRecord;
    ExceptionAddress: Pointer;
    NumberParameters: Cardinal;
    case {IsOsException:} Boolean of
      True:  (ExceptionInformation : array [0..14] of NativeUInt);
      False: (ExceptAddr: Pointer; ExceptObject: Pointer);
  end;
  TExceptClsProc = function(P: PExceptionRecord): Pointer{ExceptClass};
  TExceptObjProc = function(P: PExceptionRecord): Pointer{Exception};
  TRaiseExceptObjProc = procedure(P: PExceptionRecord);

const
  cNonContinuable     = 1;
  cUnwinding          = 2;
  cUnwindingForExit   = 4;
  cUnwindInProgress   = cUnwinding or cUnwindingForExit;
  EXCEPTION_CONTINUE_SEARCH    = 0;

type
  TAbsJump = packed record
    MovOpCode: Byte; // B8 - MOV EAX, xyz
    Ref: Pointer;
    JMP: Word;       // FF20 - JMP [EAX]
    Addr: Pointer;
  end;

const
  JumpToMemSz = SizeOf(TAbsJump);

type
  JmpInstruction =
  packed record
    opCode:   Byte;
    distance: Longint;
  end;

  PExcDescEntry = ^TExcDescEntry;
  TExcDescEntry = record
    vTable:  Pointer;
    handler: Pointer;
  end;

  PExcDesc = ^TExcDesc;
  TExcDesc = packed record
    jmp: JmpInstruction;
    case Integer of
    0:      (instructions: array [0..0] of Byte);
    1{...}: (cnt: Integer; excTab: array [0..0{cnt-1}] of TExcDescEntry);
  end;

  PExcFrame = ^TExcFrame;
  TExcFrame = record
    next: PExcFrame;
    desc: PExcDesc;
    hEBP: Pointer;
    case Integer of
    0:  ( );
    1:  ( ConstructedObject: Pointer );
    2:  ( SelfOfMethod: Pointer );
  end;

function Fix(excPtr: PExceptionRecord; errPtr: PExcFrame): PExceptionRecord;

  procedure Init;

    procedure FPUInit; assembler;
    asm
      CLD
      FNINIT
      FWAIT
    end;

  begin
    FPUInit;
    Set8087CW(Default8087CW);
  end;

var
  Rslt: TExceptionRecord;
  ExObj: TObject;
begin
  Result := excPtr;
  if (excPtr.ExceptionFlags = cUnwindInProgress) or
     (excPtr.ExceptionCode = cDelphiException) or
     (ExceptObjProc = nil) then
    Exit;

  Init;

  ExObj := TExceptObjProc(ExceptObjProc)(excPtr);

  FillChar(Rslt, SizeOf(Rslt), 0);
  Rslt.ExceptionCode := cDelphiException;
  Rslt.ExceptionFlags := cNonContinuable;
  Rslt.NumberParameters := 7;
  Rslt.ExceptAddr := excPtr^.ExceptionAddress;
  Rslt.ExceptObject := ExObj;

  Move(Rslt, excPtr^, SizeOf(Rslt));
  Result := excPtr;
end;

procedure _FpuInit;
asm
        FNINIT
        FWAIT
{$IFDEF PIC}
        CALL    GetGOT
        MOV     EAX,[EAX].OFFSET Default8087CW
        FLDCW   [EAX]
{$ELSE}
        FLDCW   Default8087CW
{$ENDIF}
end;

procedure FixedHandleAutoException; assembler;
asm
  MOV   EAX,[ESP+4]
  MOV   EDX,[ESP+8]
  CALL  FIX

        { ->    [ESP+ 4] excPtr: PExceptionRecord       }
        {       [ESP+ 8] errPtr: PExcFrame              }
        {       [ESP+12] ctxPtr: Pointer                }
        {       [ESP+16] dspPtr: Pointer                }
        { <-    EAX return value - always one           }

        MOV     EAX,[ESP+4]
        TEST    [EAX].TExceptionRecord.ExceptionFlags,cUnwindInProgress
        JNE     @@exit

        CMP     [EAX].TExceptionRecord.ExceptionCode,cDelphiException
        CLD
        CALL    _FpuInit
        JE      @@DelphiException
        CMP     BYTE PTR JITEnable,0
        JBE     @@DelphiException
        CMP     BYTE PTR DebugHook,0
        JA      @@DelphiException

@@DoUnhandled:
        LEA     EAX,[ESP+4]
        PUSH    EAX
        CALL    UnhandledExceptionFilter
        CMP     EAX,EXCEPTION_CONTINUE_SEARCH
        JE      @@exit
        MOV     EAX,[ESP+4]
        JMP     @@GoUnwind

@@DelphiException:
        CMP     BYTE PTR JITEnable,1
        JBE     @@GoUnwind
        CMP     BYTE PTR DebugHook,0
        JA      @@GoUnwind
        JMP     @@DoUnhandled

@@GoUnwind:
        OR      [EAX].TExceptionRecord.ExceptionFlags,cUnwinding

        PUSH    ESI
        PUSH    EDI
        PUSH    EBP

        MOV     EDX,[ESP+8+3*4]

        PUSH    0
        PUSH    EAX
        PUSH    offset @@returnAddress
        PUSH    EDX
        CALL    RtlUnwindProc

@@returnAddress:
        POP     EBP
        POP     EDI
        POP     ESI
        MOV     EAX,[ESP+4]
        MOV     EBX,8000FFFFH
        CMP     [EAX].TExceptionRecord.ExceptionCode,cDelphiException
        JNE     @@done

        MOV     EDX,[EAX].TExceptionRecord.ExceptObject
        MOV     ECX,[EAX].TExceptionRecord.ExceptAddr
        MOV     EAX,[ESP+8]
        MOV     EAX,[EAX].TExcFrame.SelfOfMethod
        TEST    EAX,EAX
        JZ      @@freeException
        MOV     EBX,[EAX]
        CALL    DWORD PTR [EBX] + VMTOFFSET TObject.SafeCallException
        MOV     EBX,EAX
@@freeException:
        MOV     EAX,[ESP+4]
        MOV     EAX,[EAX].TExceptionRecord.ExceptObject
        CALL    TObject.Free
@@done:
        XOR     EAX,EAX
        MOV     ESP,[ESP+8]
        POP     ECX
        MOV     FS:[EAX],ECX
        POP     EDX
        POP     EBP
        LEA     EDX,[EDX].TExcDesc.instructions
        POP     ECX
        JMP     EDX
@@exit:
        MOV     EAX,1
end;

function GetHandleAutoExceptionPointer: Pointer; assembler;
asm
  LEA EAX, System.@HandleAutoException
end;

procedure JumpToMem(const AAddr, AJump: Pointer);
var
  JumpOpCode: TAbsJump;
begin
  JumpOpCode.MovOpCode := $B8; // MOV EAX, xyz
  JumpOpCode.Ref := Pointer(NativeUInt(AAddr) + Cardinal(SizeOf(JumpOpCode.MovOpCode) + SizeOf(JumpOpCode.Ref) + SizeOf(JumpOpCode.JMP)));
  JumpOpCode.JMP := $20FF; // FF20 - JMP [EAX]
  JumpOpCode.Addr := AJump;
  Move(JumpOpCode, AAddr^, SizeOf(JumpOpCode));
end;

procedure FixSafeCallExceptions;
var
  P: Pointer;
  OldProtectionCode: DWORD;
begin
  P := GetHandleAutoExceptionPointer;
  if VirtualProtect(P, JumpToMemSz, PAGE_EXECUTE_READWRITE, @OldProtectionCode) then
  try
    JumpToMem(P, @FixedHandleAutoException);
  finally
    VirtualProtect(P, JumpToMemSz, OldProtectionCode, @OldProtectionCode);
  end;
  FlushInstructionCache(GetCurrentProcess, P, JumpToMemSz);
end;
{$ENDIF}

function ObjDescr(const AObj: TObject): String;
begin
  if AObj = nil then
  begin
    Result := 'nil';
    Exit;
  end;

  if AObj.InheritsFrom(TBaseObject) then
  begin
    Result := String(TBaseObject(AObj).DebugName);
    if Result <> '' then
      Exit;
  end;

  if AObj.InheritsFrom(TComponent) then
  begin
    Result := AObj.ClassName;
    if Result <> '' then
      Result := Format('[%s] %s', [Result, TComponent(AObj).Name]);
    if Result <> '' then
      Exit;
  end;

  Result := AObj.ClassName;
  Result := Format({$IFDEF CPU32}'[%s] $%.8x'{$ENDIF}{$IFDEF CPU64}'[%s] $%.16x'{$ENDIF}, [Result, NativeUInt(AObj)]);
end;

function MethodDescr(const AMethod): String;
var
  M: TMethod;
begin
  Move(AMethod, M, SizeOf(M));

  Result := TObject(M.Data).MethodName(M.Code);
  if Result <> '' then
    Result := Format({$IFDEF CPU32}'$%.8x (possible %s method)'{$ENDIF}{$IFDEF CPU64}'$%.16x (possible %s method)'{$ENDIF}, [NativeUInt(M.Code), Result])
  else
    Result := '$' + IntToHex(NativeUInt(M.Code), SizeOf(Pointer) * 2);
  Result := ObjDescr(TObject(M.Data)) + ' ' + Result;
end;

{ TNotify }

constructor TNotify.Create(const ANotify: TNotifyEvent);
begin
  inherited Create;
  SetName('Notify wrapper for ' + MethodDescr(ANotify));

  FNotify := ANotify;
end;

procedure TNotify.Notify;
begin
  FNotify(nil);
end;

initialization
  SafeCallErrorProc := RaiseSafeCallException;
  LoadFuncs;
  {$IFDEF WIN32}FixSafeCallExceptions;{$ENDIF}
  ReportMemoryLeaksOnShutdown := True;
end.

