unit VirtualMM;

interface

{$I VirtualMMDefs.inc}

type
  // Block protection flags
  TVirtualProtect = (
    vpNoAccess,   // block all access to the block
    vpReadOnly,   // allow reads from the block
    vpReadWrite); // full access to the block

// Changes permissions for the allocated block
// The block must be allocated via us (Delphi RTL)
procedure VirtualMMProtect(const ABlock: Pointer; const AProtect: TVirtualProtect);

implementation

uses
  Windows;

type
  {$IFDEF HAS_64BIT}
  // XE2+
  PtrInt        = NativeInt;
  PtrUInt       = NativeUInt;
  PtrIntAlloc   = PtrInt;
  {$ELSE ~HAS_64BIT}
  // XE and earlier
  PtrInt        = Integer;
  PtrUInt       = Cardinal;
  PtrIntAlloc   = PtrUInt;
  {$ENDIF HAS_64BIT}
  PPtrInt       = ^PtrInt;
  PPtrUInt      = ^PtrUInt;
  PCardinal     = ^Cardinal;

  PBlockInfo = ^TBlockInfo;
  {$IFDEF USE_SMALL_BLOCKS}
  PSmallBlockPool = ^TSmallBlockPool;

  // Data structre for a pool of small blocks
  TSmallBlockPool = record
    Flags: DWord;                  // = set of FLAG_POOL_* consts
    GuardValue: DWord;             // = GuardValue global const
    PoolStart: PBlockInfo;         // = AllocateLargeBlock
    FreeBlocks: PtrUInt;           // Number of free slots for small blocks in the .PoolStart
    Index: PtrUInt;                // Index of this pool relative to the SmallBlockPool
    ThreadID: DWORD;               // ID of a thread allocated the pool
  end;
  {$ENDIF}

  // Data structure for the large block header
  TBlockInfo = record
    GuardValue: DWord;             // = GuardValue global const
    Flags: DWord;                  // Set of FLAG_BLOCK_* consts

    {$IFDEF USE_SMALL_BLOCKS}
    PoolIndex: PtrUInt;            // Offset (starting at SmallBlockPool) to the small block pool which contains this block
    {$ENDIF}

    UnderflowPage: Pointer;        // Pointer to second page (PAGE_NOACCESS)
    BlockPage: Pointer;            // Pointer to third page (PAGE_READWRITE)
    BlockStart: Pointer;           // Pointer to data start within third page, could be equal to .BlockPage
    OverflowPage: Pointer;         // Pointer to last page (PAGE_NOACCESS)

    BlockSize: PtrInt;             // Size of block's data in bytes
    BlockPagesSize: PtrInt;        // Size of block's data in bytes rounded up to PageSize
    PagesSize: PtrInt;             // Size of all pages combined of this data
    BlockState: Integer;           // State of this block, should be bsAllocated
    ThreadID: DWORD;               // ID of a thread allocated or released the block

    // could be up to PageSize
    // another TBlockInfo follows this block when .Flags has the FLAG_BLOCK_POOL flag
  end;

const
  // .GuardValue
  GuardValue  = $DEA0BEEF;

  // .Flags
  {$IFDEF USE_SMALL_BLOCKS}
  FLAG_BLOCK_LAST = 1;             // This block is the last one in a pool, no more blocks after it
  FLAG_BLOCK_POOL = 2;             // This is a small blocks pool
  {$ENDIF}

  // .BlockState
  bsAllocated = 1;                 // GetMem (allocated)
  bsReleased  = 2;                 // FreeMem (deallocated)

var
  CS: TRTLCriticalSection;
  PageSize: PtrUInt;               // = SystemInfo.dwPageSize (4k by default)
  AllocationGranularity: PtrUInt;  // = SystemInfo.dwAllocationGranularity (64k by default)

function GetLastErrorMessage: String;
var
  ErrorCode: DWORD;
  Buffer: PChar;
  Len: Integer;
  Flags: DWORD;
begin
  ErrorCode := GetLastError;

  Flags := FORMAT_MESSAGE_FROM_SYSTEM or
           FORMAT_MESSAGE_IGNORE_INSERTS or
           FORMAT_MESSAGE_ARGUMENT_ARRAY or
           FORMAT_MESSAGE_ALLOCATE_BUFFER;

  Len := FormatMessage(Flags, nil, ErrorCode, 0, @Buffer, 0, nil);
  try
    while (Len > 0) and ((Buffer[Len - 1] <= #32) or (Buffer[Len - 1] = '.')) do
      Dec(Len);
    SetString(Result, Buffer, Len);
  finally
    LocalFree(HLOCAL(Buffer));
  end;
end;

function IsDebuggerPresent: BOOL; stdcall; external kernel32 name 'IsDebuggerPresent'; // Do Not Localize
{$EXTERNALSYM IsDebuggerPresent}

procedure FatalError(const AMsg: String);
begin
  OutputDebugString(PChar('VirtualMM: ' + AMsg));
  if IsDebuggerPresent then
    DebugBreak;

  MessageBox(0, PChar(AMsg), PChar('VirtualMM'), MB_OK or MB_ICONERROR or MB_SYSTEMMODAL or MB_SETFOREGROUND);
  TerminateProcess(GetCurrentProcess, ERROR_GEN_FAILURE);
end;

function BlockInfoToBlock(const ABlockInfo: PBlockInfo): Pointer;
// Structure: [BlockInfo] - [Guard page] - [Block's data] - [Guard page]
//            .BlockStart------------------/\
begin
  Assert(Assigned(ABlockInfo), 'BlockInfoToBlock: Assigned(ABlockInfo)');
  Assert(ABlockInfo.GuardValue = GuardValue, 'BlockInfoToBlock: ABlockInfo.GuardValue = GuardValue');
  Assert(ABlockInfo.BlockState = bsAllocated, 'BlockInfoToBlock: ABlockInfo.BlockState = bsAllocated');
  Assert(ABlockInfo.BlockSize > 0, 'BlockInfoToBlock: ABlockInfo.BlockSize > 0');
  Result := ABlockInfo.BlockStart;
end;

{$IFDEF USE_SMALL_BLOCKS}
var
  MaxSmallBlocks: PtrUInt;         // Maximum small blocks that could fit into single pool (13 by default)
  MaxSmallBlockSize: PtrUInt;      // = PageSize - 2 * VirtualMMAlign
{$ENDIF}

function BlockToBlockInfo(const ABlock: Pointer): PBlockInfo;
// Structure: [BlockInfo] - [Guard page] - [Block's data] - [Guard page]
//            [/\ allocation granularity                                 ...]
{$IFDEF USE_SMALL_BLOCKS}
var
  X: PtrUInt;
{$ENDIF}
begin
  Assert(Assigned(ABlock), 'BlockToBlockInfo: Assigned(ABlock)');

  // Align to allocation granularity, as allocation always starts with PBlockInfo
  Result := PBlockInfo(((PtrUInt(ABlock) div PtrUInt(AllocationGranularity))) * PtrUInt(AllocationGranularity));
  Assert(Result.GuardValue = GuardValue, 'BlockToBlockInfo: ABlockInfo.GuardValue = GuardValue');
  Assert(Assigned(Result.UnderflowPage), 'BlockToBlockInfo: Assigned(Result.UnderflowPage)');
  Assert(Assigned(Result.BlockPage), 'BlockToBlockInfo: Assigned(Result.BlockPage)');
  Assert(Assigned(Result.BlockStart), 'BlockToBlockInfo: Assigned(Result.BlockStart)');
  Assert(Assigned(Result.OverflowPage), 'BlockToBlockInfo: Assigned(Result.OverflowPage)');
  Assert(Result.BlockPagesSize > 0, 'BlockToBlockInfo: Result.BlockPagesSize > 0');

  {$IFDEF USE_SMALL_BLOCKS}
  if (Result.Flags and FLAG_BLOCK_POOL) <> 0 then
  begin
    // Small block - starts with pool, search it for the block
    for X := 0 to MaxSmallBlocks - 1 do
    begin
      Assert(Result.GuardValue = GuardValue, 'BlockToBlockInfo: Result.GuardValue = GuardValue');
      Assert((Result.Flags and FLAG_BLOCK_POOL) <> 0, 'BlockToBlockInfo: (Result.Flags and FLAG_BLOCK_POOL) <> 0');
      if Result.BlockStart = ABlock then
      begin
        Assert(Result.BlockState = bsAllocated, 'BlockToBlockInfo: Result.BlockState = bsAllocated');
        Assert(Result.BlockSize > 0, 'BlockToBlockInfo: Result.BlockSize > 0');
        Assert(PtrUInt(Result.BlockSize) < MaxSmallBlockSize, 'BlockToBlockInfo: Result.BlockSize < MaxSmallBlockSize');
        Exit;
      end;
      Result := PBlockInfo(PtrUInt(Result) + PtrUInt(SizeOf(TBlockInfo)));
    end;
    Assert(False, 'BlockToBlockInfo: Small Block not found');
  end
  else
  {$ENDIF}
  begin
    // Large block - always starts with its own header, return it
    Assert(Result.BlockStart = ABlock, 'BlockToBlockInfo: Result.BlockStart = ABlock');
    Assert(Result.BlockState = bsAllocated, 'BlockToBlockInfo: Result.BlockState = bsAllocated');
    Assert(Result.BlockSize > 0, 'BlockToBlockInfo: Result.BlockSize > 0');
  end;
end;

procedure VirtualMMProtect(const ABlock: Pointer; const AProtect: TVirtualProtect);
var
  BlockInfo: PBlockInfo;
  Dummy: Cardinal;
  OK: Boolean;
const
  Flags: array[TVirtualProtect] of Cardinal = (PAGE_NOACCESS, PAGE_READONLY, PAGE_READWRITE);
begin
  Assert(Assigned(ABlock), 'VirtualMMProtect: Assigned(Block)');

  BlockInfo := BlockToBlockInfo(ABlock);
  OK := VirtualProtect(BlockInfo.BlockPage, BlockInfo.BlockPagesSize, Flags[AProtect], Dummy);
  if not OK then Assert(False, 'VirtualMMProtect: [VirtualProtect] ' + GetLastErrorMessage);
end;

function Align(const P: Pointer): Pointer;
begin
  Result := Pointer((PtrUInt(P) div VirtualMMAlign) * VirtualMMAlign);
end;

function PageOffset(const P: Pointer; const Offset: PtrInt): Pointer;
var
  Pages: PtrInt;
begin
  Pages := PtrInt(PtrUInt(P) div PtrUInt(PageSize));
  Pages := PtrInt(Pages) + Offset;
  Result := Pointer(PtrUInt(Pages) * PtrUInt(PageSize));
end;

function GetAllocSizeInPages(const ASize: PtrInt): PtrInt;
// Structure: [BlockInfo] - [Guard page] - [Block's data] - [Guard page]
//            1             2                               3
begin
  Result := PtrUInt(ASize) div PageSize;
  // 4: one page for remaining data +
  // 3: one page per each side of the block (2) + one for the BlockInfo record
  if PtrUInt(ASize) mod PageSize > 0 then
    Inc(Result, 4)
  else
    Inc(Result, 3);
end;

function AllocateLargeBlock(const ASize: PtrInt): PBlockInfo;
// Structure: [BlockInfo] - [Guard page] - [Block's data] - [Guard page]
var
  Pages: PtrInt;
  Dummy: Cardinal;
  OK: Boolean;
begin
  Pages := GetAllocSizeInPages(ASize);
  Result := VirtualAlloc(nil, PtrUInt(Pages) * PageSize, MEM_RESERVE OR MEM_COMMIT {$IFDEF ALLOCATE_TOP_DOWN}OR MEM_TOP_DOWN{$ENDIF}, PAGE_READWRITE);
  if Result = nil then
    Exit;

  FillChar(Result^, Pages * PtrInt(PageSize), $CC);
  Result.GuardValue := GuardValue;
  Result.Flags := 0;
  Result.UnderflowPage := PageOffset(Result, 1); // second page
  Result.BlockPage := PageOffset(Result, 2);     // third page
  {$IFDEF PROTECT_RANDOM}
  // Random - select underflow/overflow at random
  if Random(100) < 50 then
  begin
  {$ENDIF PROTECT_RANDOM}
  {$IFDEF PROTECT_UNDERFLOW}
    // Underflow - move block down to underflow guard page
    Result.BlockStart := Result.BlockPage;
  {$ENDIF PROTECT_UNDERFLOW}
  {$IFDEF PROTECT_RANDOM}
  end
  else
  begin
  {$ENDIF PROTECT_RANDOM}
  {$IFDEF PROTECT_OVERFLOW}
    // Overflow - move block up to overflow guard page
    Dummy := PtrUInt(ASize) mod PtrUInt(PageSize); // remaining data
    if Dummy <> 0 then
      Dummy := PageSize - Dummy;
    Result.BlockStart := Pointer(PtrUInt(Result.BlockPage) + Dummy);
    Result.BlockStart := Align(Result.BlockStart);
  {$ENDIF PROTECT_OVERFLOW}
  {$IFDEF PROTECT_RANDOM}
  end;
  {$ENDIF PROTECT_RANDOM}
  Result.OverflowPage := PageOffset(Result, Pages - 1); // last page
  Result.BlockSize := ASize;
  Result.BlockPagesSize := (Pages - 3) * PtrInt(PageSize); // - one page for BlockInfo, one for underflow, one for overflow
  Result.PagesSize := Pages * PtrInt(PageSize);
  Result.BlockState := bsAllocated;
  Result.ThreadID := GetCurrentThreadId;

  OK := VirtualProtect(Result.UnderflowPage, PageSize, PAGE_NOACCESS, Dummy);
  if not OK then Assert(False, 'AllocateLargeBlock: [VirtualProtect] ' + GetLastErrorMessage);
  OK := VirtualProtect(Result.OverflowPage,  PageSize, PAGE_NOACCESS, Dummy);
  if not OK then Assert(False, 'AllocateLargeBlock: [VirtualProtect] ' + GetLastErrorMessage);
end;

{$IFDEF USE_SMALL_BLOCKS}
const
  // .Flags
  FLAG_POOL_LAST = 1;              // This pool is the last one, no more pools after it

  HEAP_NO_SERIALIZE        = 1;
  HEAP_GENERATE_EXCEPTIONS = 4;
  HEAP_ZERO_MEMORY         = 8;

var
  GSmallBlockPool: PSmallBlockPool; // Pointer to an array of TSmallBlockPool
  SmallBlockPoolSize: PtrUInt;     // Size of SmallBlockPool in bytes

// Returns pointer to the SmallBlockPool
function SmallBlockPool: PSmallBlockPool; overload;
var
  LastPool: PtrUInt;
  PoolIndex: PtrUInt;
begin
  if GSmallBlockPool <> nil then
  begin
    Result := GSmallBlockPool;
    Exit;
  end;

  // Allocate pool array/list
  SmallBlockPoolSize := AllocationGranularity;
  GSmallBlockPool := VirtualAlloc(nil, SmallBlockPoolSize, MEM_RESERVE or MEM_COMMIT, PAGE_READWRITE);
  if GSmallBlockPool = nil then Assert(False, 'GetSmallBlockPool: [VirtualAlloc] ' + GetLastErrorMessage);

  // Initialize each pool in the array/list
  LastPool := PtrUInt(GSmallBlockPool) + SmallBlockPoolSize - PtrUInt(SizeOf(TSmallBlockPool)) + 1;
  Result := GSmallBlockPool;
  PoolIndex := 0;
  while PtrUInt(Result) < LastPool do
  begin
    Result.Flags := 0;
    Result.GuardValue := GuardValue;
    Result.PoolStart := nil;
    Result.FreeBlocks := MaxSmallBlocks;
    Result.Index := PoolIndex;
    Result.ThreadID := GetCurrentThreadId;
    Result := PSmallBlockPool(PtrUInt(Result) + PtrUInt(SizeOf(TSmallBlockPool)));
    Inc(PoolIndex);
  end;

  // Mark end of pool array/list with the flag
  Result := PSmallBlockPool(PtrUInt(Result) - PtrUInt(SizeOf(TSmallBlockPool)));
  Result.Flags := FLAG_POOL_LAST;

  Result := GSmallBlockPool;
end;

function SmallBlockPool(const APoolIndex: PtrUInt): PSmallBlockPool; overload;
begin
  Result := PSmallBlockPool(PtrUInt(SmallBlockPool) + APoolIndex * PtrUInt(SizeOf(TSmallBlockPool)));
end;

// Returns APool.PoolStart (late initialization)
function GetPoolStart(const APool: PSmallBlockPool): PBlockInfo;
var
  X: PtrUInt;
begin
  if APool.PoolStart <> nil then
  begin
    Result := APool.PoolStart;
    Exit;
  end;

  APool.PoolStart := AllocateLargeBlock(APool.FreeBlocks * PageSize);

  // Initialize each small block
  Result := APool.PoolStart;
  for X := 0 to MaxSmallBlocks - 1 do
  begin
    Result.GuardValue := GuardValue;
    Result.Flags := FLAG_BLOCK_POOL;
    Result.PoolIndex := APool.Index;
    Result.UnderflowPage := APool.PoolStart.UnderflowPage;
    Result.BlockPage := Pointer(PtrUInt(APool.PoolStart.BlockPage) + X * PageSize);
    Result.OverflowPage := APool.PoolStart.OverflowPage;
    Result.BlockSize := 0;
    Result.BlockPagesSize := PageSize;
    Result.PagesSize := APool.PoolStart.PagesSize;
    Result.BlockState := bsReleased;
    Result.ThreadID := GetCurrentThreadID;

    // Small blocks are not protected with guard pages,
    // so PROTECT_RANDOM/PROTECT_UNDERFLOW/PROTECT_OVERFLOW have no effect
    // Always start small blocks at page's start (just include a guard value)
    Result.BlockStart := Pointer(PtrUInt(Result.BlockPage) + VirtualMMAlign);
    FillChar(Result.BlockPage^, PageSize, $CC);
    PCardinal(PtrUInt(Result.BlockStart) - PtrUInt(SizeOf(Cardinal)))^ := GuardValue;

    Result := PBlockInfo(PtrUInt(Result) + SizeOf(TBlockInfo));
  end;
  Result.Flags := FLAG_BLOCK_POOL or FLAG_BLOCK_LAST;

  Result := APool.PoolStart;
end;

function EnlargePool: PSmallBlockPool;
var
  OldSize: PtrUInt;
  EnlargeBy: PtrUInt;
  LastPool: PtrUInt;
  PoolIndex: PtrUInt;
  Pool: PSmallBlockPool;
begin
  EnlargeBy := AllocationGranularity;

  // Enlarge pool list/array
  SmallBlockPool;
  OldSize := SmallBlockPoolSize;
  SmallBlockPoolSize := SmallBlockPoolSize + EnlargeBy;
  Pool := VirtualAlloc(nil, SmallBlockPoolSize, MEM_RESERVE or MEM_COMMIT, PAGE_READWRITE);
  if Pool = nil then Assert(False, 'EnlargePool: [VirtualAlloc] ' + GetLastErrorMessage);
  Move(GSmallBlockPool^, Pool^, OldSize);
  if not VirtualFree(GSmallBlockPool, 0, MEM_RELEASE) then Assert(False, 'EnlargePool: [VirtualFree] ' + GetLastErrorMessage);
  GSmallBlockPool := Pool;

  // Clear "end of pool list/array" mark
  PoolIndex := OldSize div SizeOf(TSmallBlockPool) - 1; // index of the last pool in list/array
  Result := SmallBlockPool(PoolIndex);
  Assert((Result.Flags and FLAG_POOL_LAST) <> 0, 'EnlargePool: (Result.Flags and FLAG_POOL_LAST) <> 0');
  Result.Flags := Result.Flags and (not FLAG_POOL_LAST);
  Inc(PoolIndex);

  // Initialize list/array tail
  LastPool := PtrUInt(GSmallBlockPool) + SmallBlockPoolSize - PtrUInt(SizeOf(TSmallBlockPool)) + 1;
  Result := SmallBlockPool(PoolIndex);
  Pool := Result;
  while PtrUInt(Pool) < LastPool do
  begin
    Pool.Flags := 0;
    Pool.GuardValue := GuardValue;
    Pool.PoolStart := nil;
    Pool.FreeBlocks := MaxSmallBlocks;
    Pool.Index := PoolIndex;
    Pool.ThreadID := GetCurrentThreadId;
    Pool := PSmallBlockPool(PtrUInt(Pool) + PtrUInt(SizeOf(TSmallBlockPool)));
    Inc(PoolIndex);
  end;

  // Mark end of pool array/list with the flag
  Pool := PSmallBlockPool(PtrUInt(Pool) - PtrUInt(SizeOf(TSmallBlockPool)));
  Pool.Flags := FLAG_POOL_LAST;
end;

function FindPoolWithFreeSlot: PSmallBlockPool;
begin
  // Scroll through pool list until:
  // - a free slot (FreeBlocks > 0) found
  // - or end of list (FLAG_POOL_LAST is set)
  Result := SmallBlockPool;
  while (Result.GuardValue = GuardValue) and
        (Result.FreeBlocks = 0) and
        ((Result.Flags and FLAG_POOL_LAST) = 0) do
    Result := PSmallBlockPool(PtrUInt(Result) + PtrUInt(SizeOf(TSmallBlockPool)));

  // Reached end of the list?
  // Enlarge pool when there is no free space left
  if (Result.GuardValue = GuardValue) and
     (Result.FreeBlocks = 0) and
     ((Result.Flags and FLAG_POOL_LAST) <> 0) then
    Result := EnlargePool;
end;

function AllocateSmallBlock(const ASize: PtrInt): PBlockInfo;
// Small blocks structure:
//  [Pool (Heap)] -> Large Block:
//    [BlockInfo BlockInfo BlockInfo...]
//    [Block Block Block...]
var
  Pool: PSmallBlockPool;
  Block: PBlockInfo;
  X: PtrUInt;
begin
  Assert(PtrUInt(ASize) < MaxSmallBlockSize, 'AllocateSmallBlock: ASize < MaxSmallBlockSize');

  EnterCriticalSection(CS);
  try
    // Find a pool with a free slot (FreeItems should be > 0)
    Pool := FindPoolWithFreeSlot;
    Assert(Pool.FreeBlocks <> 0, 'AllocateSmallBlock: Pool.FreeItems <> 0');

    // Find a free slot in the pool
    Block := GetPoolStart(Pool);
    for X := 0 to MaxSmallBlocks - 1 do
    begin
      // Block Info should not be corrupted:
      Assert(Block.GuardValue = GuardValue, 'AllocateSmallBlock: Block.GuardValue = GuardValue');
      // Block Info should not be corrupted (pool flag MUST be set):
      Assert((Block.Flags and FLAG_BLOCK_POOL) <> 0, 'AllocateSmallBlock: (Block.Flags and FLAG_BLOCK_POOL) <> 0');
      // Block should not be corrupted (each small block starts with a guard value):
      Assert(PCardinal(PtrUInt(Block.BlockStart) - PtrUInt(SizeOf(Cardinal)))^ = GuardValue, 'AllocateSmallBlock: PCardinal(PtrUInt(Block.BlockStart) - PtrUInt(SizeOf(Cardinal)))^ = GuardValue');
      // Pool should match
      Assert(SmallBlockPool(Block.PoolIndex) = Pool, 'AllocateSmallBlock: Block.Pool = Pool');

      // Found a free slot
      if Block.BlockState = bsReleased then
      begin
        Dec(Pool.FreeBlocks);
        Block.BlockSize := ASize;
        Block.BlockState := bsAllocated;
        Block.ThreadID := GetCurrentThreadId;
        PCardinal(PtrUInt(Block.BlockStart) + PtrUInt(ASize))^ := GuardValue;
        Result := Block;
        Exit;
      end;
      Block := PBlockInfo(PtrUInt(Block) + SizeOf(TBlockInfo));
    end;
  finally
    LeaveCriticalSection(CS);
  end;
  Assert(False, 'AllocateSmallBlock: Pool.FreeItems = 0');
  Result := nil;
end;
{$ENDIF}

function VirtualGetMem(ASize: PtrInt): Pointer;
var
  ABlock: Pointer;
begin
  Assert(ASize > 0, 'VirtualGetMem: ASize > 0');

  {$IFDEF USE_SMALL_BLOCKS}
  if PtrUInt(ASize) < MaxSmallBlockSize then
    ABlock := AllocateSmallBlock(ASize)
  else
  {$ENDIF}
    ABlock := AllocateLargeBlock(ASize);

  if ABlock = nil then
    Result := nil
  else
  begin
    Result := BlockInfoToBlock(ABlock);
    Assert(Assigned(Result), 'VirtualGetMem: Assigned(Result)');
  end;
end;

{$IFNDEF UNICODE}
function InterlockedCompareExchange(var Destination: Integer; Exchange: Integer; Comperand: Integer): Integer; stdcall; external kernel32 name 'InterlockedCompareExchange'; // Do Not Localize
{$ENDIF}

{$IFDEF USE_SMALL_BLOCKS}
procedure ReleaseSmallBlock(const ABlock: Pointer; const ABlockInfo: PBlockInfo);
var
  Pool: PSmallBlockPool;
begin
  // Any exceptions (access violation or assert) in this area would indicate
  // the ABlock was already released (or corrupted)
  Assert(ABlockInfo.BlockState = bsAllocated, 'ReleaseSmallBlock: BlockInfo.BlockState = bsAllocated');
  Assert(ABlockInfo.BlockSize > 0, 'ReleaseSmallBlock: ABlockInfo.BlockSize > 0');
  Assert(PtrUInt(ABlockInfo.BlockSize) < MaxSmallBlockSize, 'ReleaseSmallBlock: ABlockInfo.BlockSize < MaxSmallBlockSize');
  Assert(PCardinal(PtrUInt(ABlockInfo.BlockStart) - PtrUInt(SizeOf(Cardinal)))^ = GuardValue, 'ReleaseSmallBlock: Block Underflow');
  Assert(PCardinal(PtrUInt(ABlockInfo.BlockStart) + PtrUInt(ABlockInfo.BlockSize))^ = GuardValue, 'ReleaseSmallBlock: Block Overflow');

  EnterCriticalSection(CS);
  try
    Pool := SmallBlockPool(ABlockInfo.PoolIndex);
    Assert(Pool.GuardValue = GuardValue, 'ReleaseSmallBlock: Pool.GuardValue = GuardValue');
    Assert(Pool.FreeBlocks < MaxSmallBlocks, 'ReleaseSmallBlock: Pool.FreeBlocks <= MaxSmallBlocks');

    ABlockInfo.BlockState := bsReleased;
    ABlockInfo.ThreadID := GetCurrentThreadId;
    ABlockInfo.BlockSize := 0;
    FillChar(ABlockInfo.BlockPage^, PageSize, $CC);
    PCardinal(PtrUInt(ABlockInfo.BlockStart) - PtrUInt(SizeOf(Cardinal)))^ := GuardValue;
    Inc(Pool.FreeBlocks);
  finally
    LeaveCriticalSection(CS);
  end;
end;
{$ENDIF}

procedure ReleaseLargeBlock(const ABlock: Pointer; const ABlockInfo: PBlockInfo);
var
  OK: Boolean;
begin
  // Any exceptions (access violation or assert) in this area would indicate
  // the ABlock was already released (or corrupted)
  if InterlockedCompareExchange(ABlockInfo.BlockState, bsReleased, bsAllocated) <> bsAllocated then
    Assert(False, 'ReleaseLargeBlock: BlockInfo.BlockState = bsAllocated');
  Assert(ABlockInfo.BlockSize > 0, 'ReleaseLargeBlock: ABlockInfo.BlockSize > 0');

  OK := VirtualFree(ABlockInfo, 0, {$IFDEF CATCH_USE_AFTER_FREE}MEM_DECOMMIT{$ELSE}MEM_RELEASE{$ENDIF});
  if not OK then Assert(False, 'ReleaseLargeBlock: [VirtualFree] ' + GetLastErrorMessage);
end;

function VirtualFreeMem(ABlock: Pointer): Integer;
var
  BlockInfo: PBlockInfo;
begin
  Assert(Assigned(ABlock), 'VirtualFreeMem: Assigned(ABlock)');

  // Any exceptions (access violation or assert) in this area would indicate
  // the ABlock was already released (or corrupted)
  BlockInfo := BlockToBlockInfo(ABlock);
  Assert(Assigned(BlockInfo), 'VirtualFreeMem: Assigned(BlockInfo)');
  {$IFDEF USE_SMALL_BLOCKS}
  if (BlockInfo.Flags and FLAG_BLOCK_POOL) <> 0 then
    ReleaseSmallBlock(ABlock, BlockInfo)
  else
  {$ENDIF}
    ReleaseLargeBlock(ABlock, BlockInfo);
  Result := 0;
end;

function VirtualAllocMem(ASize: PtrIntAlloc): Pointer;
begin
  Assert(ASize > 0, 'VirtualAllocMem: ASize > 0');
  Result := VirtualGetMem(PtrInt(ASize));
  Assert(Assigned(Result), 'VirtualAllocMem: Assigned(Result)');
  FillChar(Result^, ASize, 0);
end;

function VirtualReallocMem(ABlock: Pointer; ASize: PtrInt): Pointer;
var
  BlockInfo: PBlockInfo;
begin
  Assert(Assigned(ABlock), 'VirtualReallocMem: Assigned(ABlock)');
  Assert(PtrInt(ASize) > 0, 'VirtualReallocMem: PtrInt(ASize) > 0');

  BlockInfo := BlockToBlockInfo(ABlock);
  if BlockInfo = nil then
  begin
    Result := nil;
    Exit;
  end;

  // Always change pointer on realloc, never reallocate "in place"
  if BlockInfo.BlockSize < PtrInt(ASize) then
  begin
    // Buffer grows
    Result := VirtualAllocMem(PtrInt(ASize));
    if Result = nil then
      Exit;
    Move(ABlock^, Result^, BlockInfo.BlockSize);
    VirtualFreeMem(ABlock);
  end
  else
  begin
    // Buffer shrinks
    Result := VirtualGetMem(PtrInt(ASize));
    if Result = nil then
      Exit;
    Move(ABlock^, Result^, PtrInt(ASize));
    VirtualFreeMem(ABlock);
  end;
end;

procedure InitVirtualMM;
var
  SystemInfo: TSystemInfo;
  {$IFDEF USE_SMALL_BLOCKS}
  MaxPages: PtrUInt;
  {$ENDIF}
begin
  InitializeCriticalSection(CS);

  {$WARNINGS OFF}
  Assert(GetHeapStatus.TotalAllocated = 0, 'InitVirtualMM: Memory was already allocated before VirtualMM''s initialization');
  {$WARNINGS ON}

  if (VirtualMMAlign < 8) or (VirtualMMAlign mod 8 <> 0) then
    FatalError('InitVirtualMM: VirtualMMAlign must be disisible by 8');

  FillChar(SystemInfo, SizeOf(SystemInfo), 0);
  GetSystemInfo(SystemInfo);
  PageSize := SystemInfo.dwPageSize;
  AllocationGranularity := SystemInfo.dwAllocationGranularity;

  if VirtualMMAlign > PageSize div 4 then
    FatalError('InitVirtualMM: VirtualMMAlign is too large');

  {$IFDEF USE_SMALL_BLOCKS}
  MaxSmallBlockSize := PageSize - 2 * VirtualMMAlign;
  MaxSmallBlocks := PageSize div SizeOf(TBlockInfo); // how many headers can fit into a single page
  // When you allocate a large block of MaxPages size:
  // it must fit into AllocationGranularity exactly.
  // So, get AllocationGranularity and subtract all service info used by large blocks -
  // that would be 3 pages in total:
  // [BlockInfo] - [Guard page] - [Block's data] - [Guard page]
  // 1             2                               3
  MaxPages := (AllocationGranularity div PageSize) - 3; // = 13 for default page size of 4k and allocation granularity of 64k
  if MaxSmallBlocks > MaxPages then // True for default page size of 4k and allocation granularity of 64k
    MaxSmallBlocks := MaxPages; // = 13 by default
  {$ENDIF}
end;

procedure DoneVirtualMM;
begin
  EnterCriticalSection(CS);
  try
    SmallBlockPoolSize := 0;
    if GSmallBlockPool <> nil then
    begin
      VirtualFree(GSmallBlockPool, 0, MEM_RELEASE);
      GSmallBlockPool := nil;
    end;
  finally
    LeaveCriticalSection(CS);
    DeleteCriticalSection(CS);
  end;
end;

{$IFNDEF HAS_MEMORYMANAGER_EX}
type
  TMemoryManagerEx = TMemoryManager;
{$ENDIF}

const
  GVirtualMM: TMemoryManagerEx = (
    // The basic (required) memory manager functionality
    GetMem: VirtualGetMem;
    FreeMem: VirtualFreeMem;
    ReallocMem: VirtualReallocMem;

    // Extended (optional) functionality
    {$IFDEF HAS_MEMORYMANAGER_EX}
    AllocMem: VirtualAllocMem;
    RegisterExpectedMemoryLeak: nil;
    UnregisterExpectedMemoryLeak: nil;
    {$ENDIF}
  );

procedure InstallVirtualMM;
begin
  {$WARNINGS OFF}
  Assert(GetHeapStatus.TotalAllocated = 0, 'InstallVirtualMM: Memory was already allocated before VirtualMM''s initialization');
  {$WARNINGS ON}

  SetMemoryManager(GVirtualMM);
end;

{$IFNDEF NeverUninstall}
procedure Fail;
begin
  FatalError('Attempt to use memory manager after finalization.');
end;
{$ENDIF}

procedure UninstallVirtualMM;
{$IFNDEF NeverUninstall}
var
  FailMM: TMemoryManagerEx;
{$ENDIF}
begin
  {$IFNDEF NeverUninstall}
  @FailMM.GetMem := @Fail;
  @FailMM.FreeMem := @Fail;
  @FailMM.ReallocMem := @Fail;
  {$IFDEF HAS_MEMORYMANAGER_EX}
  @FailMM.AllocMem := @Fail;
  @FailMM.RegisterExpectedMemoryLeak := @Fail;
  @FailMM.UnregisterExpectedMemoryLeak := @Fail;
  {$ENDIF}

  SetMemoryManager(FailMM);
  {$ENDIF}
end;

initialization
  InitVirtualMM;
  InstallVirtualMM;

finalization
  UninstallVirtualMM;
  DoneVirtualMM;
end.
